function [til_u,u,til_theta,til_p,p] = TOADtbLocByRBRs_GTRSMPR(r,b,C,Q_r,Q_b)
% [til_u,u,til_theta,til_p,p] = TOADtbLocByRBRs_GTRSMPR(r,b,C,Q_r,Q_b)
%
% Distributed localization solution in MPR by GTRS using arrival time measurements 
% at multiple rigid body receivers without orientations and synchronizations.
%
% Input:
%   r:          (NML x 1), distance measurements (arrival time measurements times signal propagation speed), 
%                N = number of transmitters, M = number of RBRs,  L = number of sensors on each RBR
%   b:          (3M x 1), RBR position measurments
%   C:          (3 x L), sensor positions local to the each RBR
%   Q_r:        (NM x NM), covariance matrix of distance measurements
%   Q_b:        (3M x 3M), covariance matrix of RBR position measurments
%
% Output:
%   til_u:      (3 x 1), object position estimate in MPR
%   u:          (3 x 1), object position estimate converted to CR
%   til_theta:  (6M+N+2 x 1), complete unknown vector estimate in MPR
%   til_p:      (3M x 1), local object position estimate in MPR
%   p:          (3M x 1), local object position estimate converted to CR
%
% Reference: X. Ke and K. C. Ho, "Wireless localization in modified polar 
% representation by rigid body receivers of unknown orientations without 
% synchronizations and transmitter positions," IEEE Trans. Wireless Commun., 2025.
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

M = length(b)/3;      % number of RBRs
[~,L] = size(C);      % number of sensors on each RBR
N = length(r)/(M*L);  % number of transmitters

% -- scale the covariance matrices --
scal = Q_r(1,1);
Q_r = Q_r/scal;
Q_b = Q_b/scal;

% -- local estimation --
x = zeros(N*M,1);
delta_b = zeros(M,1);
phi = zeros(2*M,1);
Q_x = zeros(M*N,M*N);
til_p = zeros(3*M,1);
p = zeros(3*M,1);
for m = 1:1:M
    r_m = r((m-1)*L*N+1:m*L*N);               
    Q_rm = Q_r((m-1)*L*N+1:m*L*N,(m-1)*L*N+1:m*L*N);
    theta_m = LocalEst_GTRSMPR(r_m,C,Q_rm);  % GTRS local estimate
    til_p((m-1)*2+1:2*m) = theta_m(1:2);
    til_p(2*M+m) = theta_m(3);
    p((m-1)*3+1:3*m) = [cos(theta_m(1))*cos(theta_m(2));sin(theta_m(1))*cos(theta_m(2));sin(theta_m(2))]/abs(theta_m(3)); 
    phi(2*(m-1)+1:2*m)=theta_m(1:2);
    x((m-1)*N+1:m*N) = theta_m(3:N+2);        % local estimate of inverse ranges and transmitter range offsets
    delta_b(m) = theta_m(end);
    
    gm = x((m-1)*N+1);
    Jrm_deltat = kron(ones(L,1),[zeros(1,N-1);eye(N-1)]);
    rho_pm = [cos(theta_m(1))*cos(theta_m(2));sin(theta_m(1))*cos(theta_m(2));sin(theta_m(2))];
    Jrm_pm = [kron( ((rho_pm-C*gm)')./sqrt( diag((rho_pm-C*gm)'*(rho_pm-C*gm)) ), ones(N,1)), Jrm_deltat, ones(N*L,1)];
    til_pm = theta_m(1:3);
    K_pm = -[sin(til_pm(1))/cos(til_pm(2)),-cos(til_pm(1))/cos(til_pm(2)),0;
        cos(til_pm(1))*sin(til_pm(2)),sin(til_pm(1))*sin(til_pm(2)),-cos(til_pm(2));
        til_pm(3)*cos(til_pm(1))*cos(til_pm(2)),til_pm(3)*sin(til_pm(1))*cos(til_pm(2)),til_pm(3)*sin(til_pm(2))];
    K_thetam = blkdiag(K_pm,eye(N));
    Q_tilthetam = K_thetam/(Jrm_pm'/Q_rm*Jrm_pm)*K_thetam';
    Q_x((m-1)*N+1:m*N,(m-1)*N+1:m*N) = Q_tilthetam(3:N+2,3:N+2); % approximated covariance matrice of local estimate
end

% -- global estimation --
[til_u,delta_t] = GlobalEst_GTRSMPR(x,b,Q_x,Q_b); % GTRS local estimate

u = [cos(til_u(1))*cos(til_u(2));sin(til_u(1))*cos(til_u(2));sin(til_u(2))]/til_u(3);
til_theta = [til_u;phi;delta_t*til_u(3);delta_b*til_u(3);b*til_u(3)];

end

%% ================================================================================
function [theta_m] = LocalEst_GTRSMPR(r_m,C_m,Q_rm)
% [theta_m] = LocalEst_GTRSMPR(r_m,C,Q_rm)
% This function produces the local GTRS estimate in MPR.
%
% Input:
%   r_m:      (NL_m x 1), distance measurements from a RBR
%              N: number of transmitters, L_m: number of sensors 
%   C_m:      (3 x L_m), sensor positions local to the RBR
%   Q_rm:     (NL_m x NL_m), covariance matrix of distance measurements
%
% Output:
%   theta_m:  (N+3 x 1), local estimate of the complete unknown vector
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

[K,L_m] = size(C_m); % K: localization dimension, L_m: number of sensors
N = length(r_m)/L_m; % number of transmitters
P_m = kron([-ones(L_m-1,1),eye(L_m-1)],eye(N));
bar_rm = P_m*r_m;
h_m = -2*bar_rm;
bar_Cm = C_m(:,2:end) - C_m(:,1);
G_rhopm = 2*kron(bar_Cm',ones(N,1));
G_xm = bar_rm.^2-kron(diag(bar_Cm'*bar_Cm),ones(N,1));
G_m = [G_rhopm,G_xm];
Wm_inv = eye((L_m-1)*N);
Wm_sq = Wm_inv^(-0.5);
for iter = 1:1:3
    bar_Gxm = Wm_sq*G_xm;
    P_orth = eye((L_m-1)*N) - bar_Gxm*bar_Gxm'/(bar_Gxm'*bar_Gxm);
    Grhopm_par = P_orth*Wm_sq*G_rhopm;
    hm_par = P_orth*Wm_sq*h_m;
    
    % -- compute the 6th-order polynomial coefficients
    [U,Sig,V] = svd(Grhopm_par);
    sig = diag(Sig(1:K,1:K));
    a = (U(:,1:K)'*hm_par).^2.*sig.^2;
    sig2 = sig.^2;
    c0 = [1;sum(sig2);sig2(1)*sig2(2)+sig2(1)*sig2(3)+sig2(2)*sig2(3);sig2(1)*sig2(2)*sig2(3)];
    c1 = [1;sig2(2)+sig2(3);sig2(2)*sig2(3)];
    c2 = [1;sig2(1)+sig2(3);sig2(1)*sig2(3)];
    c3 = [1;sig2(2)+sig2(1);sig2(2)*sig2(1)];
    C0 = c0*c0'; C1 = c1*c1'; C2 = c2*c2'; C3 = c3*c3';
    p6 = C0(1,1);
    p5 = 2*C0(1,2);
    p4 = C0(2,2)+2*C0(1,3) - a(1)*C1(1,1) - a(2)*C2(1,1) - a(3)*C3(1,1);
    p3 = 2*(C0(1,4)+C0(2,3)) - 2*a(1)*C1(1,2) - 2*a(2)*C2(1,2)- 2*a(3)*C3(1,2);
    p2 = C0(3,3)+2*C0(2,4) - a(1)*(2*C1(1,3)+C1(2,2))- a(2)*(2*C2(1,3)+C2(2,2))- a(3)*(2*C3(1,3)+C3(2,2));
    p1 = 2*C0(3,4) - 2*a(1)*C1(2,3) - 2*a(2)*C2(2,3)- 2*a(3)*C3(2,3);
    p0 = C0(4,4) - a(1)*C1(3,3)- a(2)*C2(3,3)- a(3)*C3(3,3);
    polycoef = [p6;p5;p4;p3;p2;p1;p0];
    
    Lambda = real(roots(polycoef));
    NumSol = length(Lambda);
    alp_m = zeros(K+1,NumSol);
    CostVal = zeros(NumSol,1);
    for ii = 1:1:NumSol
        lambda = Lambda(ii);
        rho_pm = real(V*diag(1./(sig+lambda./sig))*U(:,1:K)'*hm_par);
        rho_pm = rho_pm/norm(rho_pm);
        g_m = real((bar_Gxm'*bar_Gxm)\bar_Gxm'*Wm_sq*(h_m-G_rhopm*rho_pm));
        alp_m(:,ii) = [rho_pm;g_m];
        CostVal(ii) = (h_m-G_m*alp_m(:,ii))'/Wm_inv*(h_m-G_m*alp_m(:,ii));
    end
    [~,idx] = min(CostVal);
    B_m = 2*diag(bar_rm*alp_m(4,idx)+1);
    Wm_inv = B_m*P_m*Q_rm*P_m'*B_m';
    Wm_sq = Wm_inv^(-0.5);
end
rho_pm = real(alp_m(1:3,idx));
g_m = real(alp_m(4,idx));
theta_m = [atan2(rho_pm(2),rho_pm(1));asin(rho_pm(3)/norm(rho_pm)); g_m;r_m(2:N)-r_m(1);r_m(1)*g_m-1];

end

%% ================================================================================
function [til_u,delta_t] = GlobalEst_GTRSMPR(x,b,Q_x,Q_b)
% [beta] = GlobalEst_GTRSMPR(x,b,Q_x,Q_b)
% This function produces the global GTRS estimate in MPR.
%
% Input:
%   x:         (MN x 1), local estimate
%               N: number of transmitters, M: number ofRBRs 
%   b:         (3M x 1), RBR position measurements
%   Q_x:       (MN x NM), covariance matrix of local estimate
%   Q_b:       (3M x 3M), covariance matrix of RBR position measurements 
%
% Output:
%   til_u:     (3 x 1), global estimate of object position in MPR
%   delta_t:   (N-1 x 1), estimate of transmitter range offsets
%
% Reference: X. Ke and K. C. Ho, "Wireless localization in modified polar 
% representation by rigid body receivers of unknown orientations without 
% synchronizations and transmitter positions," IEEE Trans. Wireless Commun., 2025.
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

M = length(b)/3;
N = length(x)/M;
B = reshape(b,[3,M]);
f = x;
H_beta1 = zeros(N*M,4);
H_beta2 = zeros(N*M,N-1);
for m = 1:1:M
    Idx_m = (m-1)*N+1:m*N; 
    f(Idx_m(1)) = x(Idx_m(1))^2;
    H_beta1(Idx_m(1),:) = [2*x(Idx_m(1))^2*B(:,m)',1 - x(Idx_m(1))^2*norm(B(:,m))^2];
    if N>=2
        H_beta2(Idx_m(2:N),:) = eye(N-1);
    end
end
W_inv = eye(M*N);
if N>=2
    Hbeta1_par = (eye(M*N) -H_beta2/(H_beta2'/W_inv*H_beta2)*H_beta2'/W_inv)*H_beta1;
    f_par = (eye(M*N) -H_beta2/(H_beta2'/W_inv*H_beta2)*H_beta2'/W_inv)*f;
else
    f_par = f;
    Hbeta1_par = H_beta1;
end


for iter=1:1:2
    Lambda = diag([ones(3,1);0]);
    xi = [zeros(3,1);0.5];
    A = Hbeta1_par'/W_inv*Hbeta1_par;
    if sum(find(isnan(A)))+sum(find(isinf(A)))>=1
        beta_1 = NaN*ones(4,1);
    else
        [U,Sig] = eig(A);
        T = U*sqrt(Sig);
        iT = diag(1./sqrt(diag(Sig)))*U';
        til_Lambda = iT*Lambda;
        if sum(find(isnan(til_Lambda)))+sum(find(isinf(til_Lambda)))>=1
            beta_1 = NaN*ones(4,1);
        else
            [tU,Sig,~] = svd(til_Lambda);
            sig = diag(Sig*Sig');
            [~,Indx] = sort(sig);
            sig = sig(Indx);
            U = tU(:,Indx);
            
            P = T*U;
            til_f = P\Hbeta1_par'/W_inv*f_par;
            til_xi = P\xi;
            
            
            Num = [til_xi,til_f]';
            Den = [sig,ones(4,1)]';
            c1 = PolyMulCoef(Num(:,2),Den(:,3)); c1 = PolyMulCoef(c1,Den(:,4));
            c2 = PolyMulCoef(Num(:,3),Den(:,2)); c2 = PolyMulCoef(c2,Den(:,4));
            c3 = PolyMulCoef(Num(:,4),Den(:,2)); c3 = PolyMulCoef(c3,Den(:,3));
            c0 = PolyMulCoef(Den(:,2),Den(:,3)); c0 = PolyMulCoef(c0,Den(:,4));
            cc0 = PolyMulCoef(c0,c0);
            p1 = PolyMulCoef(c1,c1)*sig(2)+PolyMulCoef(c2,c2)*sig(3)+PolyMulCoef(c3,c3)*sig(4);
            p2 = -2*(PolyMulCoef(c1,c0)*til_xi(2)+PolyMulCoef(c2,c0)*til_xi(3)+PolyMulCoef(c3,c0)*til_xi(4));
            p3 = -2*til_xi(1)*PolyMulCoef(Num(:,1),cc0);
            polycoef = [p3(1);p1+p2+p3(2:end)];
            if sum(find(isnan(polycoef)))+sum(find(isinf(polycoef)))>=1
                beta_1 = NaN*ones(4,1);
            else
                lambda_rec = real(roots(polycoef));
                NumSol = length(lambda_rec);
                beta1_rec = zeros(4,NumSol);
                CostVal = zeros(NumSol,1);
                for ii = 1:1:NumSol
                    lambda = lambda_rec(ii);
                    beta1_rec(:,ii) = (P')\real((til_f+lambda*til_xi)./(1+lambda*sig));
                    CostVal(ii) = (f_par-Hbeta1_par*beta1_rec(:,ii))'/W_inv*(f_par-Hbeta1_par*beta1_rec(:,ii));
                end
                [~,idx] = min(CostVal);
                beta_1 = real(beta1_rec(:,idx));
            end
        end
    end
    if iter==1
        D_x = eye(N*M,N*M);
        D_b = zeros(N*M,3*M);
        for m=1:1:M
            b_m = b(3*(m-1)+1:3*m);
            D_x(N*(m-1)+1,N*(m-1)+1) = 2*norm(beta_1(1:3))^2;
            D_b((m-1)*N+1,3*(m-1)+1:3*m) =2*x((m-1)*N+1)^2*(beta_1(1:3)-norm(beta_1(1:3))^2*b_m)';
        end
        W_inv = D_x*Q_x*D_x'+D_b*Q_b*D_b';
    end
end
til_u = [atan2(beta_1(2),beta_1(1));asin(beta_1(3)/norm(beta_1(1:3)));norm(beta_1(1:3))];
delta_t = (H_beta2'/W_inv*H_beta2)\H_beta2'/W_inv*(f-H_beta1*[beta_1(1:3);norm(beta_1(1:3))^2]);
end



%% ================================================================================
function Coef = PolyMulCoef(Coef_1,Coef_2)
% Coef = polyMulCoef(Coef_1,Coef_2)
% This function finds the polynomial coefficients of the product of two polynomials
%
% Input:
%   Coef_1:   (L1 x 1), coefficients of the first polynomial
%   Coef_2:   (L2 x 1), coefficients of the second polynomial
%
% Output:
%   Coef:     (L1+L2-1 x 1), coefficients of the product of two polynomials
%
% Reference: X. Ke and K. C. Ho, "Wireless localization in modified polar 
% representation by rigid body receivers of unknown orientations without 
% synchronizations and transmitter positions," IEEE Trans. Wireless Commun., 2025.
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

L1 = length(Coef_1);  
L2 = length(Coef_2);
L = max(L1,L2);
Coef_1 = [zeros(L-L1,1);Coef_1];
Coef_2 = [zeros(L-L2,1);Coef_2];
CoefProd_Mat = Coef_1*Coef_2';
Coef = zeros(2*L-1,1);
for i=1:1:L
    for j = 1:1:L
        Coef(i+j-1) = Coef(i+j-1)+CoefProd_Mat(i,j);
    end
end
Coef = Coef((2*L-1)-(L1+L2-1)+1:end);

end

