function [til_u,u] = TOAJntLocByRBRs_MLECFSMPR(r,b,C,Q_r,Q_b,MaxIter,mu,tiltheta0)
% [til_u,u] = TOAJntLocByRBRs_MLECFSMPR(r,b,C,Q_r,Q_b,MaxIter,mu,tiltheta0)
%
% Gauss-Newton MLE-MPR solution for joint localization in MPR using arrival time measurements to 
% multiple rigid body receivers without orientations and synchronizations.
%
% Input:
%   r:          (NML x 1), distance measurements (arrival time measurements times signal propagation speed), 
%                N = number of transmitters, M = number of RBRs,  L = number of sensors on each RBR
%   b:          (3M x 1), RBR position measurments
%   C:          (3 x L), sensor positions local to the each RBR
%   Q_r:        (NM x NM), covariance matrix of distance measurements
%   Q_b:        (3M x 3M), covariance matrix of RBR position measurments
%   MaxIter:    (1 x 1), maximum number of iterations
%   mu:         scalar, step size for each iteration
%   tiltheta0:  (6M+N+2 x 1), initial solution value to start iterations
%
% Output:
%   til_u:      (3 x 1), object position estimate in MPR
%   u:          (3 x 1), object position estimate converted to CR
%
% Reference: X. Ke and K. C. Ho, "Wireless localization in modified polar 
% representation by rigid body receivers of unknown orientations without 
% synchronizations and transmitter positions," IEEE Trans. Wireless Commun., 2025.
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

M = length(b)/3;      % number of RBRs
[~,L] = size(C);      % number of sensors on each RBR
N = length(r)/(M*L);  % number of transmitters

% -- scale the covariance matrices --
scal = Q_r(1,1);
Q_r = Q_r/scal;
Q_b = Q_b/scal;

tiltheta = tiltheta0;
Q = blkdiag(Q_r,Q_b);
for ii = 1:1:MaxIter
    
    % -- compute the Gauss-Newton update --
    rho_u = ang2uvec(tiltheta(1:2));
    go = tiltheta(3);
    K_tilu = -1*[sin(tiltheta(1))/cos(tiltheta(2)),-cos(tiltheta(1))/cos(tiltheta(2)),0;
        cos(tiltheta(1))*sin(tiltheta(2)),sin(tiltheta(1))*sin(tiltheta(2)),-cos(tiltheta(2));
        go*rho_u'];
    K_tiltheta = blkdiag(K_tilu,eye(6*M+N-1));
    phi = tiltheta(4:3+2*M);
    godelta_t = tiltheta(4+2*M:3+2*M+N-1);
    godelta_b = tiltheta(3+2*M+N:3+3*M+N-1);
    gob = tiltheta(3+3*M+N:end);
    til_mo = zeros(N*M*L+3*M,1);
    Jr_p = zeros(N*M*L,3*M);
    Jr_deltab = zeros(N*M*L,M);
    Jr_deltat = zeros(N*M*L,N-1);
    Jp_u = zeros(3*M,3);
    Jp_phio = zeros(3*M,2*M);
    Jp_b = zeros(3*M,3*M);
    Count = 0;
    for m = 1:1:M
        phi_m = phi((m-1)*2+1:2*m,1);
        gob_m = gob((m-1)*3+1:3*m);
        rho_pm = ang2uvec(phi_m);
        Jp_u((m-1)*3+1:3*m,1:3) = rho_pm * (rho_u-gob_m)'/norm(rho_u-gob_m);
        Jp_b((m-1)*3+1:3*m,(m-1)*3+1:3*m) = -rho_pm * (rho_u-gob_m)'/norm(rho_u-gob_m);
        Jp_phio((m-1)*3+1:3*m,(m-1)*2+1:2*m) = norm(rho_u-gob_m)*[-sin(phi_m(1,1))*cos(phi_m(2,1)),-cos(phi_m(1,1))*sin(phi_m(2,1));
            cos(phi_m(1,1))*cos(phi_m(2,1)),-sin(phi_m(1,1))*sin(phi_m(2,1));
            0,cos(phi_m(2,1))];
        for i = 1:1:L
            for k = 1:N
                Count = Count+1;
                gogm = norm(rho_u-gob_m);
                Jr_p(Count,(m-1)*3+1:3*m) = (gogm*rho_pm - go*C(:,i))'/norm(gogm*rho_pm - go*C(:,i));
                til_mo(Count,1)=norm(gogm*rho_pm - go*C(:,i)) + godelta_b(m);
                Jr_deltab(Count,m) = 1;
                if k>=2
                    Jr_deltat(Count,k-1) = 1;
                    til_mo(Count,1)=norm(gogm*rho_pm - go*C(:,i)) + godelta_b(m) + godelta_t(k-1);
                end
            end
        end
    end
    G_tiltheta = [Jr_p*Jp_u,Jr_p*Jp_phio,Jr_deltat,Jr_deltab,Jr_p*Jp_b;
        zeros(3*M,3+3*M+N-1),eye(3*M)];
    til_mo(N*M*L+1:end) = gob;
    delta_tiltheta = K_tiltheta/(G_tiltheta'/Q*G_tiltheta)*G_tiltheta'/Q*(go*[r;b]-til_mo); % Gauss-Newton update
    
    if (norm(delta_tiltheta)<=0.01)
        break;
    else
        tiltheta = tiltheta + mu*delta_tiltheta;
    end
    if isnan(sum(tiltheta))
        break;
    end
end
rho_u = sign(tiltheta(3))*ang2uvec(tiltheta(1:2));
til_u = [atan2(rho_u(2),rho_u(1));asin((rho_u(3)));abs(tiltheta(3))];
u = ang2uvec(til_u(1:2))/til_u(3);
end

function [rho] = ang2uvec(phi)
% [rho] = ang2uvec(phi)
%
% This function produces a 3 x 1 unit vector using the azimuth-elevation angle pair.
%
% Inputs:
%   phi:  (2 x 1), azimuth-elevation angle pair
%
% Outputs:
%   rho:  (3 x 1), unit vector

rho = [cos(phi(1))*cos(phi(2));sin(phi(1))*cos(phi(2));sin(phi(2))];
end
