function [Cm_u,Cm_phio,Cm_go,r_o,b_o,Q_r,Q_b] = TOALocByRBRs_CRLB(u_o,Tr_o,B_o,omega_o,C,of_tro,of_bo,sigma2_r,sigma2_b)
% [crlb_u,crlb_phio,crlb_go,r_o,b_o,Qr,Qb] = TOALocByRBRs_CRLB(u_o,Tr_o,B_o,omega_o,C,of_tro,of_bo,sigma2_r,sigma2_b)
%
% This function computes the CRLBs, true measurement vectors, and covariance matrices.
%
% Input:
%   u_o:          (3 x 1), object position
%   Tr_o:         (3 x N), transmitter positions
%   B_o:          (3 x M), RBR positions
%   omega_o:      (3 x M), RBR orientation angles
%   C:            (3 x L), sensor positions local to each RBR
%   of_tro:       (1 x N), range offsets due to absence of transmitter synchronization
%   of_bo:        (1 x M), range offsets due to absence of RBR asynchronization
%   sigma2_r:     scaler, 10*log10(noise power of distance measurements)
%   sigma2_b:     scaler, 10*log10(noise power of RBR position measurements)
%
% Output:
%   Cm_u:         (3 x 3), CRLB for object position
%   Cm_phio:      (2 x 2), CRLB for azimuth-elevation angles
%   Cm_go         scaler, CRLB for inverse-range
%   r_o           (MNL x 1), true distance measurement vector
%   b_o           (3M x 1), true RBR position measurement vector
%   Q_r           (MNL x MNL), covariance matrix of distance measurements
%   Q_b           (3M x 3M), covariance matrix of RBR position measurements
%
% Reference: X. Ke and K. C. Ho, "Wireless localization in modified polar 
% representation by rigid body receivers of unknown orientations without 
% synchronizations and transmitter positions," IEEE Trans. Wireless Commun., 2025.
%
% X. Ke, K. C. Ho     10-08-2025
%
%       Copyright (C) 2025
%       Computational Intelligence Signal Processing Laboratory
%       University of Missouri
%       Columbia, MO 65211, USA.
%       hod@missouri.edu

[~,N] = size(Tr_o);
[~,M] = size(B_o);
[~,L] = size(C);
til_uo = [atan2(u_o(2),u_o(1));asin(u_o(3)/norm(u_o));1/norm(u_o)]; 
K_uo = -til_uo(3)*[sin(til_uo(1))/cos(til_uo(2)),-cos(til_uo(1))/cos(til_uo(2)),0;
    cos(til_uo(1))*sin(til_uo(2)),sin(til_uo(1))*sin(til_uo(2)),-cos(til_uo(2));
    til_uo(3)*cos(til_uo(1))*cos(til_uo(2)),til_uo(3)*sin(til_uo(1))*cos(til_uo(2)),til_uo(3)*sin(til_uo(2))];
K_thetao = blkdiag(K_uo,eye(2*M+3*(M-M)+M+3*M+N-1));
r_o = zeros(L*N*M,1);
d_o = zeros(L*N*M,1);
Jr_p = zeros(L*N*M,3*M);
Jr_deltat = zeros(L*N*M,N-1);
Jr_deltab = zeros(L*N*M,M);
Jp_u = zeros(3*M,3);
Jp_phi = zeros(3*M,2*M);
Jp_b = zeros(3*M,3*M);
b_o = zeros(3*M,1);
id = 0;
for m = 1:1:M
    bm_o = B_o(:,m);
    b_o((m-1)*3+1:3*m,1) = bm_o;
    Rm_o = Angle2RotMtx(omega_o(:,m));
    Sm_o = Rm_o*C+bm_o;
    pm_o = Rm_o'*(u_o-bm_o);
    phim_o(1,1) = atan2(pm_o(2),pm_o(1));
    phim_o(2,1) = asin(pm_o(3)/norm(pm_o));
    Jp_u((m-1)*3+1:3*m,1:3) = pm_o/norm(pm_o) * (u_o-bm_o)'/norm(u_o-bm_o);
    Jp_b((m-1)*3+1:3*m,(m-1)*3+1:3*m) = -pm_o/norm(pm_o) * (u_o-bm_o)'/norm(u_o-bm_o);
    Jp_phi((m-1)*3+1:3*m,(m-1)*2+1:2*m) = norm(u_o-bm_o)*[-sin(phim_o(1,1))*cos(phim_o(2,1)),-cos(phim_o(1,1))*sin(phim_o(2,1));
        cos(phim_o(1,1))*cos(phim_o(2,1)),-sin(phim_o(1,1))*sin(phim_o(2,1));
        0,cos(phim_o(2,1))];
    for i = 1:1:L
        for k = 1:N
            id = id+1;
            r_o(id,1) = norm(u_o-Tr_o(:,k)) + norm(u_o-Sm_o(:,i))+of_tro(k)+of_bo(m);
            d_o(id,1) = norm(u_o-Sm_o(:,i));
            Jr_p(id,(m-1)*3+1:3*m) = (pm_o-C(:,i))'/norm(pm_o-C(:,i));
            if k>=2
                Jr_deltat(id,k-1) = 1;
            end
            Jr_deltab(id,m) = 1;
        end
    end
end
Jm_theta = [Jr_p*Jp_u,Jr_p*Jp_phi,Jr_deltat,Jr_deltab,Jr_p*Jp_b;
            zeros(3*M,3+2*M+N+M-1),eye(3*M)];
bar_Qr = 0.5*diag(d_o.^2)+0.5*(d_o*d_o');
bar_Qr_temp = zeros(N*L*M);
for mid=1:1:M
    idm = N*L*(mid-1)+1:N*L*mid;
    bar_Qr_temp(idm,idm) = bar_Qr(idm,idm);
end
bar_Qr = bar_Qr_temp;
bar_Qr = N*L*M*bar_Qr/trace(bar_Qr);
bar_Qb = eye(3*M);
Q_r = 10^(sigma2_r/10)*bar_Qr;
Q_b = 10^(sigma2_b/10)*bar_Qb;
Cm_theta = inv(Jm_theta'/blkdiag(Q_r,Q_b)*Jm_theta);
Cm_tiltheta = K_thetao*Cm_theta*K_thetao';
Cm_u = Cm_theta(1:3,1:3);
Cm_phio = Cm_tiltheta(1:2,1:2);
Cm_go = Cm_tiltheta(3,3);

end
